/* global define, nmlImpl */
define([	"lib/Zoot",		"src/build/Layer", 	"src/math/Random",	"lodash",
		"lib/dev", "lib/tasks"],
function(	Z,				Layer, 				Random,				lodash,
		dev, tasks) {
	"use strict";

/** 
 *
 * NOTES:
 * We remove scale from any transform passed to simulation.
 * That way scale only affects rendering and warping but not simulation.
 *
 * Beginning of promotion to scene-level behavior that will gather suggested transforms from other behaviors
 * and mediate.
 * 
 * TODO:
 * Scaling up/down should have no effect on simulation, but that's not the case.
 *
**/

	var kAbout = "$$$/private/animal/Behavior/RigDynamicsWorld/About=RigDynamics, (c) 2016.",
		mat3 = Z.Mat3,
		arrayPush = Array.prototype.push,
		kNoAuto = { translation : false, linear : false },
		kOriginLeaf = Layer.kOriginLeaf,
		originTagRE = new RegExp(Layer.kOriginLeaf, "g");	


	function hasNan(aMat) {
		var hasNan = false;
		aMat.forEach(function (matVal) {
			if (isNaN(matVal)) 
				hasNan = true;
		});
		return hasNan;
	}

	function getMatScene_Container (args, layer) {
		var matScene_Layer = args.getLayerMatrixRelativeToScene(layer);
		if(hasNan(matScene_Layer))
		{
			return mat3.removeScale(layer.getSourceMatrixRelativeToLayer());
		}
		return mat3.removeScale(mat3.multiply(matScene_Layer, layer.getSourceMatrixRelativeToLayer()));
	}

	// Get current Dofs of all kinematic constraints so that they can be enforced if active
	function getKinematicDofs (ids, aMatLayer_Handle) {
		var dofs = [];
		lodash.forEach(ids, function (id) {
			var mi = mat3.removeScale(aMatLayer_Handle[id]);
			// DEBUG:
			// var initial = self.getDofInitial(id, handles);
			// mat3.multiply(initial, mat3().setIdentity().rotate(0.5*Math.PI), mi);
			arrayPush.apply(dofs, mi);
		});
		return dofs;
	}

	function updateConstraintStatus(kinematicIds, draggerIds, aHandles, kinematicConstraintType){
		lodash.forEach(draggerIds, function (kinematic_index, i) {
			var id = kinematicIds[kinematic_index];
			var hi = aHandles[id];
			var active = tasks.handle.internal.isDragged(hi);
			if (active){
				kinematicConstraintType[kinematic_index] = 0;
			} else {
				kinematicConstraintType[kinematic_index] = 2;
			}
		});
	}

	// Kinematic constraints are soft-enforced in rigdyn. 
	// This post-stabilize-projects kinematic DoFs back to pre-sim by type.
	function projectKinematicDofs (ids, aKinematicDofs, kinematicConstraintType, aMatLayer_Handle) {
		lodash.forEach(ids, function (id, i) {
			// TODO: use sim.activeMouseIds to ignore projections
			var hi = aMatLayer_Handle[id]; // post-sim dof
			var hi_0 = aKinematicDofs.slice(i*9,(i+1)*9); // pre-sim dof
			if(kinematicConstraintType[i] == 0){ // weld
				hi = hi_0;
			} else if (kinematicConstraintType[i] == 1){ // hinge
				hi[6] = hi_0[6];
				hi[7] = hi_0[7];
			} else if (kinematicConstraintType[i] == 2){ // free-floating
				// no-constraint so proj is no-op for now 
			} else {
				hi = hi_0; // default weld behavior
			}
			//console.logToUser("constraint types: " + kinematicConstraintType);
			// console.logToUser("hi_p : " + hi);
			// console.logToUser("hi_0 : " + hi_0);
			aMatLayer_Handle[id] = hi;
		});
	}

	function findSims (self, args) {
		var time = args.timeStack.getRootTimeSpan().t;
		// console.logToUser("Time " + time);
		args.forEachTrackItemPuppet(time, function (rootSdkLayer) { 
			var aSimPuppets = self.tSimPuppets[rootSdkLayer.privateLayer.getStageId()];

			if(!aSimPuppets){
				// If we haven't seen this one search subpuppets for dangles
				aSimPuppets = [];
				rootSdkLayer.forEachLayerBreadthFirst(function (sdkLayer) {
					var layer = sdkLayer.privateLayer;

					// skip over layers without display container
					var displayContainer = layer.getDisplayContainer();
					if (! displayContainer) return;

					// skip over Layers without warper container
					var warperContainer = layer.getWarperContainer();
					if (! warperContainer) return;

					// skip if no warping
					var warperContainerCanWarp = warperContainer.canWarp();
					if (! warperContainerCanWarp) return;

					// skip over containers without handles
					// TODO: treat rigid body dynamics in rigdyn to sim single handle puppet
					var	aHandle = layer.gatherHandleLeafArray();
					if (! aHandle || aHandle.length < 2) return;

					// skip over containers without dangle handles
					var tree = layer.getHandleTreeArray();
					if (! lodash.find(aHandle, function (handle) { return (handle.isDangle ); })) return;

					 aSimPuppets.push({
						layer : layer,
						tree : tree,
						aHandle : aHandle,
						warperContainer : warperContainer,
						name : layer.getName()
					});
				});

				self.tSimPuppets[rootSdkLayer.privateLayer.getStageId()] = aSimPuppets;	

				// finally build each sim
				aSimPuppets.forEach(function (sim) {
					prepareSim(sim, self, args);
				});
			} 
		});
	}

	function prepareSim (sim, self, args) {
		var pinStiffness = args.getParam("pin");

		var layer = sim.layer;

		var aDynamicIds = [], 
			aKinematicIds = [], 
			aKinematicConstraintType = [],
			aDraggerIds = [],
			aDofs = [],
			matScene_Container =  getMatScene_Container(args, layer),
			// FIXME: when the state moved out the of the handles, this function became obsolete.
			// if the intention is to retrieve the current state then use tasks.puppet.getHandleFrames()
			// and filter out leaf handles.  as is, this call alwas returns handles at rest
			// NOTE: At prep we want handles at rest - is this otherwise wrong or soon-to-be unsupported?
			aLeafFrames = tasks.puppet.getLeafFrames(layer.getPuppet()),
			aMatContainer_Handle = lodash.map(aLeafFrames, lodash.property("matrix"));

		var aHandle = sim.aHandle;

		var layerMotionType = 0; // default pin
		if(layer.motionMode.translation === false && layer.motionMode.linear === true) {
			layerMotionType = 1; // hinge
		} else if (layer.motionMode.translation === true && layer.motionMode.linear === true) {
			layerMotionType = 0; //TODO: for now defualt free to pin behavior but should be free --> //2; // free
		}
		// console.logToUser("layer motionMode: " + layer.motionMode.linear);

		lodash.forEach(aHandle, function (hi, i) {

			var mi = aMatContainer_Handle[i];
			arrayPush.apply(aDofs, mat3.removeScale(mi));

			if (originTagRE.test(hi.getName())) {
				aKinematicIds.push(i);
				aKinematicConstraintType.push(layerMotionType);
				// console.logToUser("origin dof: " + hi.getName());
			} 
			else if (hi.isDraggable) {
				aKinematicIds.push(i);
				aKinematicConstraintType.push(0);
				aDraggerIds.push(aKinematicIds.length-1); 
				// console.logToUser("aDraggerId: " + aDraggerIds);
				// console.logToUser("mouse dof: " + hi.getName());
			} 
			else if (hi.isFixed) {
				aKinematicIds.push(i);
				aKinematicConstraintType.push(0);
				// console.logToUser("fixed dof: " + hi.getName());
			} 
			else if (!hi.isDangle) { // default behavior for not tagged is fixed
				aKinematicIds.push(i);
				aKinematicConstraintType.push(0);
				// console.logToUser("unspecced DoF as fixed dof: " + hi.getName());
			} 

			if (hi.isDangle) {
				// TODO: we don't really use this except for pinging stiffness value below - remove soon - simplify logic
				aDynamicIds.push(i); 
				// console.logToUser("dynamic dof: " + hi.getName());
			}

		});
		
		lodash.assign(sim, {
			rigDynamics : nmlImpl.newRigDynamics(matScene_Container, sim.warperContainer, aDofs, aKinematicIds, {
				timestep : 1.0 / self.sps,
				stiffness : aHandle[aDynamicIds[0]].springStiffness,
				pinStiffness : 100.,
				iterations : 10
			}),
			kinematicIds : aKinematicIds,
			draggerIds : aDraggerIds,
			dynamicIds : aDynamicIds,
			kinematicDofs : [],
			kinematicConstraintType : aKinematicConstraintType,
			started : false
		});
	}	

	function updateSim (sim, self, args) {
		var pinStiffness = 100.,
			robustify = args.getParam("reset"),
			kAffine = tasks.dofs.type.kAffine;

		var layer = sim.layer,
			puppet = layer.getPuppet();

		var matScene_Container = getMatScene_Container(args, layer),
			aHandles = layer.gatherHandleLeafArray(),
			alpha = 0.25; // TODO: make weight hidden live param

		var tree = layer.getHandleTreeArray(),
			aIsLeaf = tree.getIsLeafArray(),
			aLeafRef = tree.getLeafRefArray();

		var aFrames = tasks.puppet.getHandleFrames(puppet),
			// note: clone matrix values so that mutation of aMatContainer_Handle does not affect aFrames
			aMatContainer_Handle = lodash.cloneDeep(
				lodash(aFrames).at(aLeafRef).map(lodash.property("matrix")).value()
			);

		// TODO: right now assumes that stiffness is constant per sim (layer); change support to per-handle stiffness	
		var freeid = sim.dynamicIds[0];
		var latestStiffness = aHandles[freeid].springStiffness;
		if(sim.stiffness  !== latestStiffness){
			sim.rigDynamics.updateSpringStiffness(latestStiffness);
	    	sim.stiffness = latestStiffness;
		}

		if(sim.pinStiffness !== pinStiffness){
			sim.rigDynamics.updatePinStiffness(pinStiffness);
			sim.pinStiffness = pinStiffness;
		}	

		if(!sim.started){
			// sim.lastMatScene_Container = mat3.clone(matScene_Container);
			sim.lastState = aMatContainer_Handle;
			sim.nextState = aMatContainer_Handle;
			sim.currentState = aMatContainer_Handle;
			sim.timeResidual = 0.0;
			sim.last_t = 0.0;
			sim.started = true;
		} 

		sim.kinematicDofs = getKinematicDofs(sim.kinematicIds, aMatContainer_Handle); // get current pre-sim DoFs of kinematics
		// console.log("kinematicDofs: " + sim.kinematicDofs);
		// console.log("kinematic Ids: " + sim.kinematicIds);
		// console.log("constraint types: " + sim.kinematicConstraintType);

		// exponential smooth on kinematic handles
		// matScene_Container = mat3.removeScale(mat3.scaleAdd(1-alpha, sim.lastMatScene_Container, alpha, matScene_Container));
		// console.log(sim.name + ":" + matScene_Container);
		
		// update constraint type stencil for draggables and others based on "activeness/inactiveness" of drag
		updateConstraintStatus(sim.kinematicIds, sim.draggerIds, aHandles, sim.kinematicConstraintType);

		// pass updated constraints to rigdyn
		sim.rigDynamics.setKinematicHandles(matScene_Container, sim.kinematicConstraintType, sim.kinematicDofs); 
		// for smoothing
		// sim.lastMatScene_Container = mat3.clone(matScene_Container);
		
		sim.rigDynamics.setBodyForce(aHandles[freeid].dangleWind[0] + aHandles[freeid].dangleGravity[0],aHandles[freeid].dangleWind[1] + aHandles[freeid].dangleGravity[1]);

		if(robustify){
			sim.rigDynamics.resetShapeState();
		}

		// step @ fixed
		sim.rigDynamics.step(aMatContainer_Handle);
		sim.currentState = aMatContainer_Handle;
		//--

		// step interpolated
		// var curr_t = Math.max(0.0,args.t) + args.globalRehearseTime; 
		// if (sim.last_t < 0.0 ) {
		// 	sim.last_t = curr_t;
		// }
		
		// // var delta_t = curr_t - sim.last_t;
		// // var delta_t = 1/args.scene_.getFrameRate();
		// // do not go backwards in time; if simulation stepping is the bottleneck do not add to residual
		// var delta_t = Math.min(Math.max(0.0,curr_t - sim.last_t), 4/args.scene_.getFrameRate());
		// sim.last_t = curr_t;

		// sim.timeResidual = sim.timeResidual + delta_t;
		// var dt = 1.0/self.sps;
		// // console.logToUser("dt : " + dt);
		
		// while (sim.timeResidual > dt){
		// 	sim.lastState = sim.nextState
		// 	sim.rigDynamics.step(aMatContainer_Handle);
		// 	sim.nextState = aMatContainer_Handle;
		// 	sim.timeResidual = sim.timeResidual - dt;
		// }
		
		// sim.timeResidual = Math.round(sim.timeResidual * 1000) / 1000;
		// dt = Math.round(dt * 1000) / 1000;
		// var beta = sim.timeResidual/dt;

		// // console.logToUser("timeResidual : " + sim.timeResidual);
		// // console.logToUser("beta :" + beta);
		// lodash.forEach(sim.currentState, function (hi, i) {
		// 	sim.currentState[i] = mat3.scaleAdd(beta, sim.nextState[i], 1-beta, sim.lastState[i]);
		// });
		//---

		aMatContainer_Handle = sim.currentState;

		// kinematic constraints are soft-enforced in rigdyn. post-stabilize-project kinematic DoFs back to pre-sim by type.
		projectKinematicDofs(sim.kinematicIds, sim.kinematicDofs, sim.kinematicConstraintType, aMatContainer_Handle);

		// update handles (at this point aMatContainer_Handle contains all handle updates)
		var leafIndex = -1;
		lodash.forEach(aFrames, function (frame, index) {
			// non-leaves are never updated
			if ( !aIsLeaf[index] ) return;
			leafIndex += 1;
			// in-place update of frame matrix and type
			frame.matrix = aMatContainer_Handle[leafIndex];
			frame.type = kAffine;
		});

		// and update puppet with new handle frames
		tasks.puppet.setHandleFrames(puppet, aFrames);
	}

	function stepSims (self, args) {
		var time = args.timeStack.getRootTimeSpan().t;
		args.forEachTrackItemPuppet(time, function (sdkLayer) { 
			var layer = sdkLayer.privateLayer;
			var aSimPuppets = self.tSimPuppets[layer.getStageId()];
			if(aSimPuppets){ 
				if(aSimPuppets.length > 0){ 
					// step sims
					aSimPuppets.forEach(function (sim) {
						updateSim(sim, self, args);
						// console.logToUser(" sim name = "  + sim.name);
					});


					layer.commitMatrixDofs();					
				}
			}
		});
	}

	return {
		about: kAbout,
		description: "$$$/animal/Behavior/RigDynamicsWorld/Desc=Physics simulation",
		uiName: 	"$$$/animal/Behavior/RigDynamicsWorld/UIName=RigDynamics",
		defaultArmedForRecordOn: true,
		applyToSceneOnly: true,
	
		defineParams: function () { 
			// free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{
					id: "pin", type: "slider", uiName: "$$$/animal/Behavior/RigDynamicsWorld/Parameter/pin=Attachment Strength",
					min:0, max:10000, precision:0, dephault: 100
				},
				{
					id: "iterations", type: "slider", uiName: "$$$/animal/Behavior/RigDynamicsWorld/Parameter/iterations=Number of Iterations",
					min:0, max:1000, precision:0, dephault: 10,
					hidden:true
				},
				{
					id: "reset", type: "checkbox", uiName: "$$$/animal/Behavior/RigDynamicsWorld/Parameter/reset=Reset", dephault: false,
					hidden:true
				}
			];
		},
		
		onCreateBackStageBehavior: function (/*self*/) {
			return {
				order : 1.0,
				importance : 1.0
			};
		},
		
		onCreateStageBehavior: function (self/*, args*/) {
			self.tSimPuppets = {};
			self.rand = new Random(123456789);
			self.sps = 12.0;
		},

		onResetRehearsalData : function (self,args) {
			lodash.forEach(self.tSimPuppets, function (aSimPuppets) {
	    		if (aSimPuppets) {
	        		lodash.forEach(aSimPuppets, function (sim) {
	            		sim.last_t = -1;
		       		 });
		   		 }
			    // otherwise, not initialized and no sim puppets
			});
		},
		
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			findSims(self, args);
			stepSims(self, args);
		}
	}; // end of object being returned
});
